/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.core.io.buffer;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.channels.AsynchronousFileChannel;
import java.nio.channels.Channel;
import java.nio.channels.Channels;
import java.nio.channels.CompletionHandler;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.reactivestreams.Publisher;
import org.reactivestreams.Subscription;
import org.springframework.core.io.Resource;
import org.springframework.core.io.buffer.DataBuffer;
import org.springframework.core.io.buffer.DataBufferFactory;
import org.springframework.core.io.buffer.LimitedDataBufferList;
import org.springframework.core.io.buffer.PooledDataBuffer;
import org.springframework.lang.Nullable;
import org.springframework.util.Assert;
import reactor.core.CoreSubscriber;
import reactor.core.Disposable;
import reactor.core.publisher.BaseSubscriber;
import reactor.core.publisher.Flux;
import reactor.core.publisher.FluxSink;
import reactor.core.publisher.Mono;
import reactor.core.publisher.MonoSink;
import reactor.core.publisher.SynchronousSink;

public abstract class DataBufferUtils {
    private static final Log logger = LogFactory.getLog(DataBufferUtils.class);
    private static final Consumer<DataBuffer> RELEASE_CONSUMER = DataBufferUtils::release;

    public static Flux<DataBuffer> readInputStream(Callable<InputStream> callable, DataBufferFactory dataBufferFactory, int n) {
        Assert.notNull(callable, "'inputStreamSupplier' must not be null");
        return DataBufferUtils.readByteChannel(() -> Channels.newChannel((InputStream)callable.call()), dataBufferFactory, n);
    }

    public static Flux<DataBuffer> readByteChannel(Callable<ReadableByteChannel> callable, DataBufferFactory dataBufferFactory, int n) {
        Assert.notNull(callable, "'channelSupplier' must not be null");
        Assert.notNull((Object)dataBufferFactory, "'dataBufferFactory' must not be null");
        Assert.isTrue(n > 0, "'bufferSize' must be > 0");
        return Flux.using(callable, readableByteChannel -> Flux.generate((Consumer)new ReadableByteChannelGenerator((ReadableByteChannel)readableByteChannel, dataBufferFactory, n)), DataBufferUtils::closeChannel);
    }

    public static Flux<DataBuffer> readAsynchronousFileChannel(Callable<AsynchronousFileChannel> callable, DataBufferFactory dataBufferFactory, int n) {
        return DataBufferUtils.readAsynchronousFileChannel(callable, 0L, dataBufferFactory, n);
    }

    public static Flux<DataBuffer> readAsynchronousFileChannel(Callable<AsynchronousFileChannel> callable, long l, DataBufferFactory dataBufferFactory, int n) {
        Assert.notNull(callable, "'channelSupplier' must not be null");
        Assert.notNull((Object)dataBufferFactory, "'dataBufferFactory' must not be null");
        Assert.isTrue(l >= 0L, "'position' must be >= 0");
        Assert.isTrue(n > 0, "'bufferSize' must be > 0");
        Flux flux = Flux.using(callable, asynchronousFileChannel -> Flux.create(fluxSink -> {
            ReadCompletionHandler readCompletionHandler = new ReadCompletionHandler((AsynchronousFileChannel)asynchronousFileChannel, (FluxSink<DataBuffer>)fluxSink, l, dataBufferFactory, n);
            fluxSink.onCancel(readCompletionHandler::cancel);
            fluxSink.onRequest(readCompletionHandler::request);
        }), asynchronousFileChannel -> {});
        return flux.doOnDiscard(PooledDataBuffer.class, DataBufferUtils::release);
    }

    public static Flux<DataBuffer> read(Path path, DataBufferFactory dataBufferFactory, int n, OpenOption ... openOptionArray) {
        Assert.notNull((Object)path, "Path must not be null");
        Assert.notNull((Object)dataBufferFactory, "BufferFactory must not be null");
        Assert.isTrue(n > 0, "'bufferSize' must be > 0");
        if (openOptionArray.length > 0) {
            for (OpenOption openOption : openOptionArray) {
                Assert.isTrue(openOption != StandardOpenOption.APPEND && openOption != StandardOpenOption.WRITE, "'" + openOption + "' not allowed");
            }
        }
        return DataBufferUtils.readAsynchronousFileChannel(() -> AsynchronousFileChannel.open(path, openOptionArray), dataBufferFactory, n);
    }

    public static Flux<DataBuffer> read(Resource resource, DataBufferFactory dataBufferFactory, int n) {
        return DataBufferUtils.read(resource, 0L, dataBufferFactory, n);
    }

    public static Flux<DataBuffer> read(Resource resource, long l, DataBufferFactory dataBufferFactory, int n) {
        try {
            if (resource.isFile()) {
                File file = resource.getFile();
                return DataBufferUtils.readAsynchronousFileChannel(() -> AsynchronousFileChannel.open(file.toPath(), StandardOpenOption.READ), l, dataBufferFactory, n);
            }
        }
        catch (IOException iOException) {
        }
        Flux<DataBuffer> flux = DataBufferUtils.readByteChannel(resource::readableChannel, dataBufferFactory, n);
        return l == 0L ? flux : DataBufferUtils.skipUntilByteCount(flux, l);
    }

    public static Flux<DataBuffer> write(Publisher<DataBuffer> publisher, OutputStream outputStream2) {
        Assert.notNull(publisher, "'source' must not be null");
        Assert.notNull((Object)outputStream2, "'outputStream' must not be null");
        WritableByteChannel writableByteChannel = Channels.newChannel(outputStream2);
        return DataBufferUtils.write(publisher, writableByteChannel);
    }

    public static Flux<DataBuffer> write(Publisher<DataBuffer> publisher, WritableByteChannel writableByteChannel) {
        Assert.notNull(publisher, "'source' must not be null");
        Assert.notNull((Object)writableByteChannel, "'channel' must not be null");
        Flux flux = Flux.from(publisher);
        return Flux.create(fluxSink -> {
            WritableByteChannelSubscriber writableByteChannelSubscriber = new WritableByteChannelSubscriber((FluxSink<DataBuffer>)fluxSink, writableByteChannel);
            fluxSink.onDispose((Disposable)writableByteChannelSubscriber);
            flux.subscribe((CoreSubscriber)writableByteChannelSubscriber);
        });
    }

    public static Flux<DataBuffer> write(Publisher<DataBuffer> publisher, AsynchronousFileChannel asynchronousFileChannel) {
        return DataBufferUtils.write(publisher, asynchronousFileChannel, 0L);
    }

    public static Flux<DataBuffer> write(Publisher<? extends DataBuffer> publisher, AsynchronousFileChannel asynchronousFileChannel, long l) {
        Assert.notNull(publisher, "'source' must not be null");
        Assert.notNull((Object)asynchronousFileChannel, "'channel' must not be null");
        Assert.isTrue(l >= 0L, "'position' must be >= 0");
        Flux flux = Flux.from(publisher);
        return Flux.create(fluxSink -> {
            WriteCompletionHandler writeCompletionHandler = new WriteCompletionHandler((FluxSink<DataBuffer>)fluxSink, asynchronousFileChannel, l);
            fluxSink.onDispose((Disposable)writeCompletionHandler);
            flux.subscribe((CoreSubscriber)writeCompletionHandler);
        });
    }

    public static Mono<Void> write(Publisher<DataBuffer> publisher, Path path, OpenOption ... openOptionArray) {
        Assert.notNull(publisher, "Source must not be null");
        Assert.notNull((Object)path, "Destination must not be null");
        Set<OpenOption> set = DataBufferUtils.checkWriteOptions(openOptionArray);
        return Mono.create(monoSink -> {
            try {
                AsynchronousFileChannel asynchronousFileChannel = AsynchronousFileChannel.open(path, set, null, new FileAttribute[0]);
                monoSink.onDispose(() -> DataBufferUtils.closeChannel(asynchronousFileChannel));
                DataBufferUtils.write(publisher, asynchronousFileChannel).subscribe(DataBufferUtils::release, arg_0 -> ((MonoSink)monoSink).error(arg_0), () -> ((MonoSink)monoSink).success());
            }
            catch (IOException iOException) {
                monoSink.error((Throwable)iOException);
            }
        });
    }

    private static Set<OpenOption> checkWriteOptions(OpenOption[] openOptionArray) {
        int n = openOptionArray.length;
        HashSet<OpenOption> hashSet = new HashSet<OpenOption>(n + 3);
        if (n == 0) {
            hashSet.add(StandardOpenOption.CREATE);
            hashSet.add(StandardOpenOption.TRUNCATE_EXISTING);
        } else {
            for (OpenOption openOption : openOptionArray) {
                if (openOption == StandardOpenOption.READ) {
                    throw new IllegalArgumentException("READ not allowed");
                }
                hashSet.add(openOption);
            }
        }
        hashSet.add(StandardOpenOption.WRITE);
        return hashSet;
    }

    static void closeChannel(@Nullable Channel channel) {
        if (channel != null && channel.isOpen()) {
            try {
                channel.close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
    }

    public static Flux<DataBuffer> takeUntilByteCount(Publisher<? extends DataBuffer> publisher, long l) {
        Assert.notNull(publisher, "Publisher must not be null");
        Assert.isTrue(l >= 0L, "'maxByteCount' must be a positive number");
        return Flux.defer(() -> {
            AtomicLong atomicLong = new AtomicLong(l);
            return Flux.from((Publisher)publisher).map(dataBuffer -> {
                long l = atomicLong.addAndGet(-dataBuffer.readableByteCount());
                if (l < 0L) {
                    int n = dataBuffer.readableByteCount() + (int)l;
                    return dataBuffer.slice(0, n);
                }
                return dataBuffer;
            }).takeUntil(dataBuffer -> atomicLong.get() <= 0L);
        });
    }

    public static Flux<DataBuffer> skipUntilByteCount(Publisher<? extends DataBuffer> publisher, long l) {
        Assert.notNull(publisher, "Publisher must not be null");
        Assert.isTrue(l >= 0L, "'maxByteCount' must be a positive number");
        return Flux.defer(() -> {
            AtomicLong atomicLong = new AtomicLong(l);
            return Flux.from((Publisher)publisher).skipUntil(dataBuffer -> {
                long l = atomicLong.addAndGet(-dataBuffer.readableByteCount());
                return l < 0L;
            }).map(dataBuffer -> {
                long l = atomicLong.get();
                if (l < 0L) {
                    atomicLong.set(0L);
                    int n = dataBuffer.readableByteCount() + (int)l;
                    int n2 = (int)(-l);
                    return dataBuffer.slice(n, n2);
                }
                return dataBuffer;
            });
        }).doOnDiscard(PooledDataBuffer.class, DataBufferUtils::release);
    }

    public static <T extends DataBuffer> T retain(T t2) {
        if (t2 instanceof PooledDataBuffer) {
            return (T)((PooledDataBuffer)t2).retain();
        }
        return t2;
    }

    public static boolean release(@Nullable DataBuffer dataBuffer) {
        PooledDataBuffer pooledDataBuffer;
        if (dataBuffer instanceof PooledDataBuffer && (pooledDataBuffer = (PooledDataBuffer)dataBuffer).isAllocated()) {
            try {
                return pooledDataBuffer.release();
            }
            catch (IllegalStateException illegalStateException) {
                if (logger.isDebugEnabled()) {
                    logger.debug("Failed to release PooledDataBuffer", illegalStateException);
                }
                return false;
            }
        }
        return false;
    }

    public static Consumer<DataBuffer> releaseConsumer() {
        return RELEASE_CONSUMER;
    }

    public static Mono<DataBuffer> join(Publisher<? extends DataBuffer> publisher) {
        return DataBufferUtils.join(publisher, -1);
    }

    public static Mono<DataBuffer> join(Publisher<? extends DataBuffer> publisher, int n) {
        Assert.notNull(publisher, "'dataBuffers' must not be null");
        if (publisher instanceof Mono) {
            return (Mono)publisher;
        }
        return Flux.from(publisher).collect(() -> new LimitedDataBufferList(n), LimitedDataBufferList::add).filter(limitedDataBufferList -> !limitedDataBufferList.isEmpty()).map(limitedDataBufferList -> ((DataBuffer)limitedDataBufferList.get(0)).factory().join((List<? extends DataBuffer>)limitedDataBufferList)).doOnDiscard(PooledDataBuffer.class, DataBufferUtils::release);
    }

    public static Matcher matcher(byte[] byArray) {
        return DataBufferUtils.createMatcher(byArray);
    }

    public static Matcher matcher(byte[] ... byArray) {
        Assert.isTrue(byArray.length > 0, "Delimiters must not be empty");
        return byArray.length == 1 ? DataBufferUtils.createMatcher(byArray[0]) : new CompositeMatcher(byArray);
    }

    private static NestedMatcher createMatcher(byte[] byArray) {
        Assert.isTrue(byArray.length > 0, "Delimiter must not be empty");
        switch (byArray.length) {
            case 1: {
                return byArray[0] == 10 ? SingleByteMatcher.NEWLINE_MATCHER : new SingleByteMatcher(byArray);
            }
            case 2: {
                return new TwoByteMatcher(byArray);
            }
        }
        return new KnuthMorrisPrattMatcher(byArray);
    }

    private static class WriteCompletionHandler
    extends BaseSubscriber<DataBuffer>
    implements CompletionHandler<Integer, ByteBuffer> {
        private final FluxSink<DataBuffer> sink;
        private final AsynchronousFileChannel channel;
        private final AtomicBoolean completed = new AtomicBoolean();
        private final AtomicReference<Throwable> error = new AtomicReference();
        private final AtomicLong position;
        private final AtomicReference<DataBuffer> dataBuffer = new AtomicReference();

        public WriteCompletionHandler(FluxSink<DataBuffer> fluxSink, AsynchronousFileChannel asynchronousFileChannel, long l) {
            this.sink = fluxSink;
            this.channel = asynchronousFileChannel;
            this.position = new AtomicLong(l);
        }

        protected void hookOnSubscribe(Subscription subscription) {
            this.request(1L);
        }

        protected void hookOnNext(DataBuffer dataBuffer) {
            if (!this.dataBuffer.compareAndSet(null, dataBuffer)) {
                throw new IllegalStateException();
            }
            ByteBuffer byteBuffer = dataBuffer.asByteBuffer();
            this.channel.write(byteBuffer, this.position.get(), byteBuffer, this);
        }

        protected void hookOnError(Throwable throwable) {
            this.error.set(throwable);
            if (this.dataBuffer.get() == null) {
                this.sink.error(throwable);
            }
        }

        protected void hookOnComplete() {
            this.completed.set(true);
            if (this.dataBuffer.get() == null) {
                this.sink.complete();
            }
        }

        @Override
        public void completed(Integer n, ByteBuffer byteBuffer) {
            long l = this.position.addAndGet(n.intValue());
            if (byteBuffer.hasRemaining()) {
                this.channel.write(byteBuffer, l, byteBuffer, this);
                return;
            }
            this.sinkDataBuffer();
            Throwable throwable = this.error.get();
            if (throwable != null) {
                this.sink.error(throwable);
            } else if (this.completed.get()) {
                this.sink.complete();
            } else {
                this.request(1L);
            }
        }

        @Override
        public void failed(Throwable throwable, ByteBuffer byteBuffer) {
            this.sinkDataBuffer();
            this.sink.error(throwable);
        }

        private void sinkDataBuffer() {
            DataBuffer dataBuffer = this.dataBuffer.get();
            Assert.state(dataBuffer != null, "DataBuffer should not be null");
            this.sink.next((Object)dataBuffer);
            this.dataBuffer.set(null);
        }
    }

    private static class WritableByteChannelSubscriber
    extends BaseSubscriber<DataBuffer> {
        private final FluxSink<DataBuffer> sink;
        private final WritableByteChannel channel;

        public WritableByteChannelSubscriber(FluxSink<DataBuffer> fluxSink, WritableByteChannel writableByteChannel) {
            this.sink = fluxSink;
            this.channel = writableByteChannel;
        }

        protected void hookOnSubscribe(Subscription subscription) {
            this.request(1L);
        }

        protected void hookOnNext(DataBuffer dataBuffer) {
            try {
                ByteBuffer byteBuffer = dataBuffer.asByteBuffer();
                while (byteBuffer.hasRemaining()) {
                    this.channel.write(byteBuffer);
                }
                this.sink.next((Object)dataBuffer);
                this.request(1L);
            }
            catch (IOException iOException) {
                this.sink.next((Object)dataBuffer);
                this.sink.error((Throwable)iOException);
            }
        }

        protected void hookOnError(Throwable throwable) {
            this.sink.error(throwable);
        }

        protected void hookOnComplete() {
            this.sink.complete();
        }
    }

    private static class ReadCompletionHandler
    implements CompletionHandler<Integer, DataBuffer> {
        private final AsynchronousFileChannel channel;
        private final FluxSink<DataBuffer> sink;
        private final DataBufferFactory dataBufferFactory;
        private final int bufferSize;
        private final AtomicLong position;
        private final AtomicReference<State> state = new AtomicReference<State>(State.IDLE);

        public ReadCompletionHandler(AsynchronousFileChannel asynchronousFileChannel, FluxSink<DataBuffer> fluxSink, long l, DataBufferFactory dataBufferFactory, int n) {
            this.channel = asynchronousFileChannel;
            this.sink = fluxSink;
            this.position = new AtomicLong(l);
            this.dataBufferFactory = dataBufferFactory;
            this.bufferSize = n;
        }

        public void request(long l) {
            this.tryRead();
        }

        public void cancel() {
            this.state.getAndSet(State.DISPOSED);
            DataBufferUtils.closeChannel(this.channel);
        }

        private void tryRead() {
            if (this.sink.requestedFromDownstream() > 0L && this.state.compareAndSet(State.IDLE, State.READING)) {
                this.read();
            }
        }

        private void read() {
            DataBuffer dataBuffer = this.dataBufferFactory.allocateBuffer(this.bufferSize);
            ByteBuffer byteBuffer = dataBuffer.asByteBuffer(0, this.bufferSize);
            this.channel.read(byteBuffer, this.position.get(), dataBuffer, this);
        }

        @Override
        public void completed(Integer n, DataBuffer dataBuffer) {
            if (this.state.get().equals((Object)State.DISPOSED)) {
                DataBufferUtils.release(dataBuffer);
                DataBufferUtils.closeChannel(this.channel);
                return;
            }
            if (n == -1) {
                DataBufferUtils.release(dataBuffer);
                DataBufferUtils.closeChannel(this.channel);
                this.state.set(State.DISPOSED);
                this.sink.complete();
                return;
            }
            this.position.addAndGet(n.intValue());
            dataBuffer.writePosition(n);
            this.sink.next((Object)dataBuffer);
            if (this.sink.requestedFromDownstream() > 0L) {
                this.read();
                return;
            }
            if (this.state.compareAndSet(State.READING, State.IDLE)) {
                this.tryRead();
            }
        }

        @Override
        public void failed(Throwable throwable, DataBuffer dataBuffer) {
            DataBufferUtils.release(dataBuffer);
            DataBufferUtils.closeChannel(this.channel);
            this.state.set(State.DISPOSED);
            this.sink.error(throwable);
        }

        private static enum State {
            IDLE,
            READING,
            DISPOSED;

        }
    }

    private static class ReadableByteChannelGenerator
    implements Consumer<SynchronousSink<DataBuffer>> {
        private final ReadableByteChannel channel;
        private final DataBufferFactory dataBufferFactory;
        private final int bufferSize;

        public ReadableByteChannelGenerator(ReadableByteChannel readableByteChannel, DataBufferFactory dataBufferFactory, int n) {
            this.channel = readableByteChannel;
            this.dataBufferFactory = dataBufferFactory;
            this.bufferSize = n;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void accept(SynchronousSink<DataBuffer> synchronousSink) {
            boolean bl = true;
            DataBuffer dataBuffer = this.dataBufferFactory.allocateBuffer(this.bufferSize);
            try {
                ByteBuffer byteBuffer = dataBuffer.asByteBuffer(0, dataBuffer.capacity());
                int n = this.channel.read(byteBuffer);
                if (n >= 0) {
                    dataBuffer.writePosition(n);
                    bl = false;
                    synchronousSink.next((Object)dataBuffer);
                } else {
                    synchronousSink.complete();
                }
            }
            catch (IOException iOException) {
                synchronousSink.error((Throwable)iOException);
            }
            finally {
                if (bl) {
                    DataBufferUtils.release(dataBuffer);
                }
            }
        }
    }

    private static class KnuthMorrisPrattMatcher
    extends AbstractNestedMatcher {
        private final int[] table;

        public KnuthMorrisPrattMatcher(byte[] byArray) {
            super(byArray);
            this.table = KnuthMorrisPrattMatcher.longestSuffixPrefixTable(byArray);
        }

        private static int[] longestSuffixPrefixTable(byte[] byArray) {
            int[] nArray = new int[byArray.length];
            nArray[0] = 0;
            for (int i = 1; i < byArray.length; ++i) {
                int n = nArray[i - 1];
                while (n > 0 && byArray[i] != byArray[n]) {
                    n = nArray[n - 1];
                }
                if (byArray[i] == byArray[n]) {
                    // empty if block
                }
                nArray[i] = ++n;
            }
            return nArray;
        }

        @Override
        public boolean match(byte by) {
            while (this.getMatches() > 0 && by != this.delimiter()[this.getMatches()]) {
                this.setMatches(this.table[this.getMatches() - 1]);
            }
            return super.match(by);
        }
    }

    private static class TwoByteMatcher
    extends AbstractNestedMatcher {
        protected TwoByteMatcher(byte[] byArray) {
            super(byArray);
            Assert.isTrue(byArray.length == 2, "Expected a 2-byte delimiter");
        }
    }

    private static abstract class AbstractNestedMatcher
    implements NestedMatcher {
        private final byte[] delimiter;
        private int matches = 0;

        protected AbstractNestedMatcher(byte[] byArray) {
            this.delimiter = byArray;
        }

        protected void setMatches(int n) {
            this.matches = n;
        }

        protected int getMatches() {
            return this.matches;
        }

        @Override
        public int match(DataBuffer dataBuffer) {
            for (int i = dataBuffer.readPosition(); i < dataBuffer.writePosition(); ++i) {
                byte by = dataBuffer.getByte(i);
                if (!this.match(by)) continue;
                this.reset();
                return i;
            }
            return -1;
        }

        @Override
        public boolean match(byte by) {
            if (by == this.delimiter[this.matches]) {
                ++this.matches;
                return this.matches == this.delimiter().length;
            }
            return false;
        }

        @Override
        public byte[] delimiter() {
            return this.delimiter;
        }

        @Override
        public void reset() {
            this.matches = 0;
        }
    }

    private static class SingleByteMatcher
    implements NestedMatcher {
        static SingleByteMatcher NEWLINE_MATCHER = new SingleByteMatcher(new byte[]{10});
        private final byte[] delimiter;

        SingleByteMatcher(byte[] byArray) {
            Assert.isTrue(byArray.length == 1, "Expected a 1 byte delimiter");
            this.delimiter = byArray;
        }

        @Override
        public int match(DataBuffer dataBuffer) {
            for (int i = dataBuffer.readPosition(); i < dataBuffer.writePosition(); ++i) {
                byte by = dataBuffer.getByte(i);
                if (!this.match(by)) continue;
                return i;
            }
            return -1;
        }

        @Override
        public boolean match(byte by) {
            return this.delimiter[0] == by;
        }

        @Override
        public byte[] delimiter() {
            return this.delimiter;
        }

        @Override
        public void reset() {
        }
    }

    private static interface NestedMatcher
    extends Matcher {
        public boolean match(byte var1);
    }

    private static class CompositeMatcher
    implements Matcher {
        private static final byte[] NO_DELIMITER = new byte[0];
        private final NestedMatcher[] matchers;
        byte[] longestDelimiter = NO_DELIMITER;

        CompositeMatcher(byte[][] byArray) {
            this.matchers = CompositeMatcher.initMatchers(byArray);
        }

        private static NestedMatcher[] initMatchers(byte[][] byArray) {
            NestedMatcher[] nestedMatcherArray = new NestedMatcher[byArray.length];
            for (int i = 0; i < byArray.length; ++i) {
                nestedMatcherArray[i] = DataBufferUtils.createMatcher(byArray[i]);
            }
            return nestedMatcherArray;
        }

        @Override
        public int match(DataBuffer dataBuffer) {
            this.longestDelimiter = NO_DELIMITER;
            for (int i = dataBuffer.readPosition(); i < dataBuffer.writePosition(); ++i) {
                byte by = dataBuffer.getByte(i);
                for (NestedMatcher nestedMatcher : this.matchers) {
                    if (!nestedMatcher.match(by) || nestedMatcher.delimiter().length <= this.longestDelimiter.length) continue;
                    this.longestDelimiter = nestedMatcher.delimiter();
                }
                if (this.longestDelimiter == NO_DELIMITER) continue;
                this.reset();
                return i;
            }
            return -1;
        }

        @Override
        public byte[] delimiter() {
            Assert.state(this.longestDelimiter != NO_DELIMITER, "Illegal state!");
            return this.longestDelimiter;
        }

        @Override
        public void reset() {
            for (NestedMatcher nestedMatcher : this.matchers) {
                nestedMatcher.reset();
            }
        }
    }

    public static interface Matcher {
        public int match(DataBuffer var1);

        public byte[] delimiter();

        public void reset();
    }
}

