/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.hash;

import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.hash.AbstractCompositeHashFunction;
import com.google.common.hash.ChecksumHashFunction;
import com.google.common.hash.Crc32cHashFunction;
import com.google.common.hash.ElementTypesAreNonnullByDefault;
import com.google.common.hash.FarmHashFingerprint64;
import com.google.common.hash.Fingerprint2011;
import com.google.common.hash.HashCode;
import com.google.common.hash.HashFunction;
import com.google.common.hash.Hasher;
import com.google.common.hash.IgnoreJRERequirement;
import com.google.common.hash.ImmutableSupplier;
import com.google.common.hash.MacHashFunction;
import com.google.common.hash.MessageDigestHashFunction;
import com.google.common.hash.Murmur3_128HashFunction;
import com.google.common.hash.Murmur3_32HashFunction;
import com.google.common.hash.SipHashFunction;
import com.google.errorprone.annotations.Immutable;
import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.lang.invoke.MethodType;
import java.security.Key;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.zip.Adler32;
import java.util.zip.CRC32;
import java.util.zip.Checksum;
import javax.annotation.CheckForNull;
import javax.crypto.spec.SecretKeySpec;

@ElementTypesAreNonnullByDefault
public final class Hashing {
    static final int GOOD_FAST_HASH_SEED = (int)System.currentTimeMillis();

    public static HashFunction goodFastHash(int n) {
        int n2 = Hashing.checkPositiveAndMakeMultipleOf32(n);
        if (n2 == 32) {
            return Murmur3_32HashFunction.GOOD_FAST_HASH_32;
        }
        if (n2 <= 128) {
            return Murmur3_128HashFunction.GOOD_FAST_HASH_128;
        }
        int n3 = (n2 + 127) / 128;
        HashFunction[] hashFunctionArray = new HashFunction[n3];
        hashFunctionArray[0] = Murmur3_128HashFunction.GOOD_FAST_HASH_128;
        int n4 = GOOD_FAST_HASH_SEED;
        for (int i = 1; i < n3; ++i) {
            hashFunctionArray[i] = Hashing.murmur3_128(n4 += 1500450271);
        }
        return new ConcatenatedHashFunction(hashFunctionArray);
    }

    @Deprecated
    public static HashFunction murmur3_32(int n) {
        return new Murmur3_32HashFunction(n, false);
    }

    @Deprecated
    public static HashFunction murmur3_32() {
        return Murmur3_32HashFunction.MURMUR3_32;
    }

    public static HashFunction murmur3_32_fixed(int n) {
        return new Murmur3_32HashFunction(n, true);
    }

    public static HashFunction murmur3_32_fixed() {
        return Murmur3_32HashFunction.MURMUR3_32_FIXED;
    }

    public static HashFunction murmur3_128(int n) {
        return new Murmur3_128HashFunction(n);
    }

    public static HashFunction murmur3_128() {
        return Murmur3_128HashFunction.MURMUR3_128;
    }

    public static HashFunction sipHash24() {
        return SipHashFunction.SIP_HASH_24;
    }

    public static HashFunction sipHash24(long l, long l2) {
        return new SipHashFunction(2, 4, l, l2);
    }

    @Deprecated
    public static HashFunction md5() {
        return Md5Holder.MD5;
    }

    @Deprecated
    public static HashFunction sha1() {
        return Sha1Holder.SHA_1;
    }

    public static HashFunction sha256() {
        return Sha256Holder.SHA_256;
    }

    public static HashFunction sha384() {
        return Sha384Holder.SHA_384;
    }

    public static HashFunction sha512() {
        return Sha512Holder.SHA_512;
    }

    public static HashFunction hmacMd5(Key key) {
        return new MacHashFunction("HmacMD5", key, Hashing.hmacToString("hmacMd5", key));
    }

    public static HashFunction hmacMd5(byte[] byArray) {
        return Hashing.hmacMd5(new SecretKeySpec(Preconditions.checkNotNull(byArray), "HmacMD5"));
    }

    public static HashFunction hmacSha1(Key key) {
        return new MacHashFunction("HmacSHA1", key, Hashing.hmacToString("hmacSha1", key));
    }

    public static HashFunction hmacSha1(byte[] byArray) {
        return Hashing.hmacSha1(new SecretKeySpec(Preconditions.checkNotNull(byArray), "HmacSHA1"));
    }

    public static HashFunction hmacSha256(Key key) {
        return new MacHashFunction("HmacSHA256", key, Hashing.hmacToString("hmacSha256", key));
    }

    public static HashFunction hmacSha256(byte[] byArray) {
        return Hashing.hmacSha256(new SecretKeySpec(Preconditions.checkNotNull(byArray), "HmacSHA256"));
    }

    public static HashFunction hmacSha512(Key key) {
        return new MacHashFunction("HmacSHA512", key, Hashing.hmacToString("hmacSha512", key));
    }

    public static HashFunction hmacSha512(byte[] byArray) {
        return Hashing.hmacSha512(new SecretKeySpec(Preconditions.checkNotNull(byArray), "HmacSHA512"));
    }

    private static String hmacToString(String string, Key key) {
        return "Hashing." + string + "(Key[algorithm=" + key.getAlgorithm() + ", format=" + key.getFormat() + "])";
    }

    public static HashFunction crc32c() {
        return Crc32CSupplier.HASH_FUNCTION;
    }

    public static HashFunction crc32() {
        return ChecksumType.CRC_32.hashFunction;
    }

    public static HashFunction adler32() {
        return ChecksumType.ADLER_32.hashFunction;
    }

    public static HashFunction farmHashFingerprint64() {
        return FarmHashFingerprint64.FARMHASH_FINGERPRINT_64;
    }

    public static HashFunction fingerprint2011() {
        return Fingerprint2011.FINGERPRINT_2011;
    }

    public static int consistentHash(HashCode hashCode, int n) {
        return Hashing.consistentHash(hashCode.padToLong(), n);
    }

    public static int consistentHash(long l, int n) {
        int n2;
        Preconditions.checkArgument(n > 0, "buckets must be positive: %s", n);
        LinearCongruentialGenerator linearCongruentialGenerator = new LinearCongruentialGenerator(l);
        int n3 = 0;
        while ((n2 = (int)((double)(n3 + 1) / linearCongruentialGenerator.nextDouble())) >= 0 && n2 < n) {
            n3 = n2;
        }
        return n3;
    }

    public static HashCode combineOrdered(Iterable<HashCode> iterable) {
        Iterator<HashCode> iterator2 = iterable.iterator();
        Preconditions.checkArgument(iterator2.hasNext(), "Must be at least 1 hash code to combine.");
        int n = iterator2.next().bits();
        byte[] byArray = new byte[n / 8];
        for (HashCode hashCode : iterable) {
            byte[] byArray2 = hashCode.asBytes();
            Preconditions.checkArgument(byArray2.length == byArray.length, "All hashcodes must have the same bit length.");
            for (int i = 0; i < byArray2.length; ++i) {
                byArray[i] = (byte)(byArray[i] * 37 ^ byArray2[i]);
            }
        }
        return HashCode.fromBytesNoCopy(byArray);
    }

    public static HashCode combineUnordered(Iterable<HashCode> iterable) {
        Iterator<HashCode> iterator2 = iterable.iterator();
        Preconditions.checkArgument(iterator2.hasNext(), "Must be at least 1 hash code to combine.");
        byte[] byArray = new byte[iterator2.next().bits() / 8];
        for (HashCode hashCode : iterable) {
            byte[] byArray2 = hashCode.asBytes();
            Preconditions.checkArgument(byArray2.length == byArray.length, "All hashcodes must have the same bit length.");
            for (int i = 0; i < byArray2.length; ++i) {
                int n = i;
                byArray[n] = (byte)(byArray[n] + byArray2[i]);
            }
        }
        return HashCode.fromBytesNoCopy(byArray);
    }

    static int checkPositiveAndMakeMultipleOf32(int n) {
        Preconditions.checkArgument(n > 0, "Number of bits must be positive");
        return n + 31 & 0xFFFFFFE0;
    }

    public static HashFunction concatenating(HashFunction hashFunction, HashFunction hashFunction2, HashFunction ... hashFunctionArray) {
        ArrayList<HashFunction> arrayList = new ArrayList<HashFunction>();
        arrayList.add(hashFunction);
        arrayList.add(hashFunction2);
        Collections.addAll(arrayList, hashFunctionArray);
        return new ConcatenatedHashFunction(arrayList.toArray(new HashFunction[0]));
    }

    public static HashFunction concatenating(Iterable<HashFunction> iterable) {
        Preconditions.checkNotNull(iterable);
        ArrayList<HashFunction> arrayList = new ArrayList<HashFunction>();
        for (HashFunction hashFunction : iterable) {
            arrayList.add(hashFunction);
        }
        Preconditions.checkArgument(!arrayList.isEmpty(), "number of hash functions (%s) must be > 0", arrayList.size());
        return new ConcatenatedHashFunction(arrayList.toArray(new HashFunction[0]));
    }

    private Hashing() {
    }

    private static final class LinearCongruentialGenerator {
        private long state;

        public LinearCongruentialGenerator(long l) {
            this.state = l;
        }

        public double nextDouble() {
            this.state = 2862933555777941757L * this.state + 1L;
            return (double)((int)(this.state >>> 33) + 1) / 2.147483648E9;
        }
    }

    private static final class ConcatenatedHashFunction
    extends AbstractCompositeHashFunction {
        private ConcatenatedHashFunction(HashFunction ... hashFunctionArray) {
            super(hashFunctionArray);
            for (HashFunction hashFunction : hashFunctionArray) {
                Preconditions.checkArgument(hashFunction.bits() % 8 == 0, "the number of bits (%s) in hashFunction (%s) must be divisible by 8", hashFunction.bits(), (Object)hashFunction);
            }
        }

        @Override
        HashCode makeHash(Hasher[] hasherArray) {
            byte[] byArray = new byte[this.bits() / 8];
            int n = 0;
            for (Hasher hasher : hasherArray) {
                HashCode hashCode = hasher.hash();
                n += hashCode.writeBytesTo(byArray, n, hashCode.bits() / 8);
            }
            return HashCode.fromBytesNoCopy(byArray);
        }

        @Override
        public int bits() {
            int n = 0;
            for (HashFunction hashFunction : this.functions) {
                n += hashFunction.bits();
            }
            return n;
        }

        public boolean equals(@CheckForNull Object object) {
            if (object instanceof ConcatenatedHashFunction) {
                ConcatenatedHashFunction concatenatedHashFunction = (ConcatenatedHashFunction)object;
                return Arrays.equals(this.functions, concatenatedHashFunction.functions);
            }
            return false;
        }

        public int hashCode() {
            return Arrays.hashCode(this.functions);
        }
    }

    private static final class Crc32cMethodHandles {
        private static final MethodHandle CONSTRUCTOR = Crc32cMethodHandles.crc32cConstructor();

        private Crc32cMethodHandles() {
        }

        @IgnoreJRERequirement
        static Checksum newCrc32c() {
            try {
                return CONSTRUCTOR.invokeExact();
            }
            catch (Throwable throwable) {
                Throwables.throwIfUnchecked(throwable);
                throw Crc32cMethodHandles.newLinkageError(throwable);
            }
        }

        private static MethodHandle crc32cConstructor() {
            try {
                Class<?> clazz = Class.forName("java.util.zip.CRC32C");
                return MethodHandles.lookup().findConstructor(clazz, MethodType.methodType(Void.TYPE)).asType(MethodType.methodType(Checksum.class));
            }
            catch (ClassNotFoundException classNotFoundException) {
                throw new AssertionError((Object)classNotFoundException);
            }
            catch (IllegalAccessException illegalAccessException) {
                throw Crc32cMethodHandles.newLinkageError(illegalAccessException);
            }
            catch (NoSuchMethodException noSuchMethodException) {
                throw Crc32cMethodHandles.newLinkageError(noSuchMethodException);
            }
        }

        private static LinkageError newLinkageError(Throwable throwable) {
            return new LinkageError(throwable.toString(), throwable);
        }
    }

    @Immutable
    static enum ChecksumType implements ImmutableSupplier<Checksum>
    {
        CRC_32("Hashing.crc32()"){

            @Override
            public Checksum get() {
                return new CRC32();
            }
        }
        ,
        CRC_32C("Hashing.crc32c()"){

            @Override
            public Checksum get() {
                return Crc32cMethodHandles.newCrc32c();
            }
        }
        ,
        ADLER_32("Hashing.adler32()"){

            @Override
            public Checksum get() {
                return new Adler32();
            }
        };

        public final HashFunction hashFunction;

        private ChecksumType(String string2) {
            this.hashFunction = new ChecksumHashFunction(this, 32, string2);
        }
    }

    @Immutable
    private static enum Crc32CSupplier implements ImmutableSupplier<HashFunction>
    {
        JAVA_UTIL_ZIP{

            @Override
            public HashFunction get() {
                return ChecksumType.CRC_32C.hashFunction;
            }
        }
        ,
        ABSTRACT_HASH_FUNCTION{

            @Override
            public HashFunction get() {
                return Crc32cHashFunction.CRC_32_C;
            }
        };

        static final HashFunction HASH_FUNCTION;

        private static Crc32CSupplier pickFunction() {
            Crc32CSupplier[] crc32CSupplierArray = Crc32CSupplier.values();
            if (crc32CSupplierArray.length == 1) {
                return crc32CSupplierArray[0];
            }
            Crc32CSupplier crc32CSupplier = crc32CSupplierArray[0];
            try {
                Class.forName("java.util.zip.CRC32C");
                return crc32CSupplier;
            }
            catch (ClassNotFoundException classNotFoundException) {
                return ABSTRACT_HASH_FUNCTION;
            }
        }

        static {
            HASH_FUNCTION = (HashFunction)Crc32CSupplier.pickFunction().get();
        }
    }

    private static class Sha512Holder {
        static final HashFunction SHA_512 = new MessageDigestHashFunction("SHA-512", "Hashing.sha512()");

        private Sha512Holder() {
        }
    }

    private static class Sha384Holder {
        static final HashFunction SHA_384 = new MessageDigestHashFunction("SHA-384", "Hashing.sha384()");

        private Sha384Holder() {
        }
    }

    private static class Sha256Holder {
        static final HashFunction SHA_256 = new MessageDigestHashFunction("SHA-256", "Hashing.sha256()");

        private Sha256Holder() {
        }
    }

    private static class Sha1Holder {
        static final HashFunction SHA_1 = new MessageDigestHashFunction("SHA-1", "Hashing.sha1()");

        private Sha1Holder() {
        }
    }

    private static class Md5Holder {
        static final HashFunction MD5 = new MessageDigestHashFunction("MD5", "Hashing.md5()");

        private Md5Holder() {
        }
    }
}

