﻿#pragma once
#include "UQMMacros.h"
#include "UQMCrash.h"

namespace GCloud {
    namespace CrashSight {
        class CRASHSIGHT_API CrashSightAgent {
            typedef void (*CrashCallbackFuncPtr)(int type, const char* guid);
        private:
            static bool is_log_path_setted_;
        public:
            /************************************全平台接口************************************/

            /**
             * @brief [EN] Initialize CrashSight
             *        [ZH] 初始化CrashSight
             * @param appId：[EN] APPID of registered projects [ZH] 已注册项目的APPID
             * @param forceOnUiThread：[EN] Whether CrashSight should be forced to initialize on the UI thread, only affects Android [ZH] 是否强制CrashSight在ui线程上初始化，仅影响安卓端
             * @note  [EN] Initializes as early as possible to enable crash detection and reporting features
             *        [ZH] 在尽可能早的位置进行初始化以开启崩溃捕获和上报功能
             */
            static void InitWithAppId(const char* app_id, bool forceOnUiThread = false);

            /**
             * @brief [EN] Actively reports errors [ZH] 主动上报错误
             * @param type [EN] Exception type, 0 ~ 3 are internal reserved types and are invalid if passed in, C#: 4, js: 5, lua: 6
             *             [ZH] 异常类型, 0 ~ 3为内部保留类型传入无效, C#: 4, js: 5, lua: 6
             * @param name [EN] Exception Name, cannot be null [ZH] 异常名称，不能为null
             * @param reason [EN] Exception reason, cannot be null [ZH] 异常原因，不能为null
             * @param stack_trace [EN] Stack trace, cannot be null [ZH] 堆栈，不能为null
             * @param extras [EN] Extra information [ZH] 其他信息
             * @param quit [EN] Whether to quit [ZH] 是否退出
             * @param dump_native_type [EN] 0: close, 1: call system interface dump, 3: minidump (only valid for mobile)
             *                         [ZH] 0：关闭，1：调用系统接口dump，3：minidump(仅移动端有效)
             * @param is_async [EN] Whether async (Win/Xbox only) [ZH] 是否异步(仅Win/Xbox有效)
             * @param error_attach_path [EN] Absolute path of the attachment (GBK characters are not supported) (valid for Win and Xbox) 
             *                          [ZH] 附件的绝对路径(不支持GBK字符)(Win、Xbox有效)
             * @note [EN] Can be called manually when an error is caught or needs to be reported, multi-threaded calls are supported. 
             *       [ZH] 可以在捕获到错误或者需要上报的时候手动调用，支持多线程调用       
             */
            static void ReportException(int type, const char* name, const char* reason, const char* stack_trace,
                                        const char* extras, bool quit, int dump_native_type = 0, bool is_async = true,
                                        const char *error_attach_path = nullptr);

            /**
             * @brief [EN] Actively reports errors (wide char version) [ZH] 主动上报错误(宽字符版本)
             * @param type [EN] Exception type, 0 ~ 3 are internal reserved types and are invalid if passed in, C#: 4, js: 5, lua: 6
             *             [ZH] 异常类型, 0 ~ 3为内部保留类型传入无效, C#: 4, js: 5, lua: 6
             * @param name [EN] Exception Name, cannot be null [ZH] 异常名称，不能为null
             * @param reason [EN] Exception reason, cannot be null [ZH] 异常原因，不能为null
             * @param stack_trace [EN] Stack trace, cannot be null [ZH] 堆栈，不能为null
             * @param extras [EN] Extra information [ZH] 其他信息
             * @param quit [EN] Whether to quit [ZH] 是否退出
             * @param dump_native_type [EN] 0: close, 1: call system interface dump, 3: minidump (only valid for mobile)
             *                         [ZH] 0：关闭，1：调用系统接口dump，3：minidump(仅移动端有效)
             * @param is_async [EN] Whether async (Win/Xbox only) [ZH] 是否异步(仅Win/Xbox有效)
             * @param error_attach_path [EN] Log attachment path (wide char), Win only [ZH] 日志附件路径(宽字符)，仅Windows端有效。
             * @note [EN] Can be called manually when an error is caught or needs to be reported, multi-threaded calls are supported.
             *            Supports paths with GBK characters. 
             *       [ZH] 可以在捕获到错误或者需要上报的时候手动调用，支持多线程调用，支持带GBK字符的路径。    
             */
            static void ReportExceptionW(int type, const char* name, const char* reason, const char* stack_trace,
                                         const char* extras, bool quit, int dump_native_type = 0, bool is_async = true,
                                         const wchar_t *error_attach_path = nullptr);

            /**
             * @brief [EN] Setting user ID
             *        [ZH] 设置用户ID
             * @param user_id [EN] User identifier [ZH] 用户标识符
             * @note [EN] The user id defaults to unknown [ZH] 用户ID默认为unknown
             */
            static void SetUserId(const char* user_id);

            /**
             * @brief [EN] Adds custom data [ZH] 添加自定义数据
             * @param key
             * @param value 
             * @note [EN] Set the Key-Value data customized by the user. It will be reported together with exception info when sending the crash.
             *            View page: Crash Details->Download Attachments->valueMapOthers.txt
             *       [ZH] 设置用户自定义的 Key-Value 数据，将在发送 Crash 时随异常信息一起上报
             *            页面查看：崩溃详情页->附件下载->valueMapOthers.txt
             */
            static void AddSceneData(const char* key, const char* value);

            /**
             * @brief [EN] Adds custom data [ZH] 添加自定义数据
             * @param key
             * @param value 
             * @note [EN] Call "AddSceneData" to set the Key-Value data customized by the user, where the value is int type.
             *            View page: Crash Details->Download Attachments->valueMapOthers.txt      
             *       [ZH] 调用 "AddSceneData",设置用户自定义的 Key-Value 数据，其中 value 为 int 类型。
             *            页面查看：崩溃详情页->附件下载->valueMapOthers.txt
             */
            static void SetUserValue(const char* key, int value);

            /**
             * @brief [EN] Adds custom data [ZH] 添加自定义数据
             * @param key
             * @param value 
             * @note [EN] Call "AddSceneData" to set the Key-Value data customized by the user, where the value is double type.
             *            View page: Crash Details->Download Attachments->valueMapOthers.txt   
             *       [ZH] 调用 "AddSceneData",设置用户自定义的 Key-Value 数据，其中 value 为 double 类型。
             *            页面查看：崩溃详情页->附件下载->valueMapOthers.txt
             */
            static void SetUserValue(const char* key, double value);

            /**
             * @brief [EN] Adds custom data [ZH] 添加自定义数据
             * @param key
             * @param value 
             * @note [EN] Call "AddSceneData" to set the Key-Value data customized by the user, where the value is char* type.
             *            View page: Crash Details->Download Attachments->valueMapOthers.txt   
             *       [ZH] 调用 "AddSceneData",设置用户自定义的 Key-Value 数据，其中 value 为 char* 类型。
             *            页面查看：崩溃详情页->附件下载->valueMapOthers.txt
             */
            static void SetUserValue(const char* key, const char* value);

            /**
             * @brief [EN] Adds custom data [ZH] 添加自定义数据
             * @param key
             * @param value [EN] String array containing multiple values [ZH] 包含多个值的字符串数组
             * @param count [EN] Number of elements in the values array [ZH] values数组中的元素数量
             * @note [EN] Call "AddSceneData" to set the Key-Value data customized by the user, where the value is a string array.
             *            View page: Crash Details->Download Attachments->valueMapOthers.txt        
             *       [ZH] 调用 "AddSceneData",设置用户自定义的 Key-Value 数据，其中 value 为字符串数组类型。
             *            页面查看：崩溃详情页->附件下载->valueMapOthers.txt
             */ 
            static void SetUserValue(const char* key, const char** values, int count);

            /**
             * @brief [EN] Sets application version [ZH] 设置应用版本号
             * @param app_version [EN] Version Number [ZH] 版本号
             * @note [EN] Need to be called before "InitWithAppId" interface
             *       [ZH] 需要在 "InitWithAppId" 接口之前调用
             */  
            static void SetAppVersion(const char* app_version);

            /**
             * @brief [EN] Sets the reporting domain name [ZH] 设置上报域名
             * @param crash_server_url [EN] Target domain name of reporting [ZH] 要上报的域名
             * @note [EN] Need to be called before "InitWithAppId" interface.
             *       [ZH] 需要在 "InitWithAppId" 接口之前调用
             */ 
            static void ConfigCrashServerUrl(const char* crash_server_url);

            /**
             * @brief [EN] Sets upload path for log after crash [ZH] 设置崩溃后上传的日志路径
             * @param log_path [EN] Absolute path for log files [ZH] 日志文件绝对路径
             * @note [EN] Read permission is required. Not available on Switch platform 
             *       [ZH] 需要可读权限, Switch平台不可用
             */
            static void SetLogPath(const char* log_path);

            /**
             * @brief [EN] Sets upload path for log after crash (Wide char version, Windows only) 
             *        [ZH] 设置崩溃后上传的日志路径(宽字符版本，仅Windows)
             * @param log_path [EN] Absolute path for log files [ZH] 日志文件绝对路径
             * @note [EN] Read permission is required. Supports paths with GBK characters.
             *       [ZH] 需要可读权限，支持带GBK字符的路径。
             */
            static void SetLogPathW(const wchar_t* log_path);

            /**
             * @brief [EN] Sets debug mode [ZH] 设置debug模式
             * @param enable 
             * @note [EN] Disabled by default. Need to be called before "InitWithAppId" interface. 
             *            When enabled, more logs will be printed to facilitate problem location.
             *       [ZH] 默认关闭, 需要在 "InitWithAppId" 接口之前调用，开启后会打印更多便于问题定位的Log
             */
            static void ConfigDebugMode(bool enable);

            /**
             * @brief [EN] Sets device id [ZH] 设置设备Id
             * @param device_id [EN] Device ID [ZH] 设备唯一标识
             * @note [EN] By default, uuid is used as the device ID. Need to be called before "InitWithAppId" interface.
             *       [ZH] 默认采用uuid作为设备ID, 需要在 "InitWithAppId" 接口之前调用
             */ 
            static void SetDeviceId(const char* device_id);

            /**
             * @brief [EN] Sets custom log reporting level [ZH] 设置自定义日志上报级别
             * @param log_level [EN] Off=0, Error=1, Warn=2, Info=3, Debug=4, default is Info 
             *                  [ZH] Off=0, Error=1, Warn=2, Info=3, Debug=4, 默认Info
             * @note [EN] Need to be called before "InitWithAppId" interface.
             *       [ZH] 需要在 "InitWithAppId" 接口之前调用
             */
            static void ConfigCrashReporter(int log_level);

            /**
             * @brief [EN] Adds custom logs [ZH] 添加自定义日志
             * @param level [EN] Log level [ZH] 日志级别
             * @param format [EN] Format string [ZH] 格式化字符串
             * @param ... [EN] Variable arguments [ZH] 可变参数
             * @note [EN] The custom log shouldn't exceed 30KB. Custom log view: 
             *            Android, iOS, PS4, PS5, Switch: issue details->tracking log->custom log
             *            Windows, Xbox, Linux: issue details -> custom log (from interface)
             *       [ZH] 自定义日志，限制30KB。 自定义日志查看：
             *            Android、iOS、PS4、PS5、Switch：问题详情->跟踪日志->custom log
             *            Windows、Xbox、Linux：问题详情->自定义日志(来自接口)
             */ 
            static void PrintLog(LogSeverity level, const char* format, ...);

            /**
             * @brief [EN] Tests native crash [ZH] 测试 native 崩溃
             */
            static void TestNativeCrash();

            /**
             * @brief [EN] Sets the distribution channel [ZH] 设置发行渠道
             * @param serverEnv
             * @note [EN] Each network connection or report can carry this field to collect statistics for different distribution channels.
             *            It is not available on PS4、PS5、Switch and Linux.
             *       [ZH] 每一个联网或者上报都可以携带该字段，可实现针对不同的发行渠道统计数据。 PS4、PS5、Switch、Linux暂不可用
             */
            static void SetEnvironmentName(const char *serverEnv);

            /**
             * @brief [EN] Sets the distribution channel [ZH] 设置发行渠道
             * @param serverEnv
             * @note [EN] Each network connection or report can carry this field to collect statistics for different distribution channels.
             *            It is not available on PS4、PS5、Switch and Linux. Recommend using the SetEnvironmentName function
             *       [ZH] 每一个联网或者上报都可以携带该字段，可实现针对不同的发行渠道统计数据。PS4、PS5、Switch、Linux暂不可用。推荐使用SetEnvironmentName函数
             */
            static void setServerEnv(const char *serverEnv);

            /**
             * @brief [EN] Sets crash callback [ZH] 设置崩溃回调
             * @param crashObserver [EN] Callback class inheriting from UQMCrashObserver [ZH] 继承UQMCrashObserver的回调类
             * @note [EN] Callback is called when a crash occurs and the return value will be included in the crash report.
             *            It is not available on PS4, PS5, Switch, Linux and HarmonyOS.
             *       [ZH] 回调在崩溃时调用，返回值随崩溃信息一起上报。PS4、PS5、Switch、Linux、鸿蒙暂不可用
             */ 
            static void SetCrashObserver(UQM::UQMCrashObserver* crashObserver);

            /**
             * @brief [EN] Sets engine information(called automatically by plugin) [ZH] 设置引擎信息(由插件自动调用)
             * @note [EN] Not available on PS4/PS5/Switch/Linux [ZH] PS4/PS5/Switch/Linux平台不可用
             */ 
            static void SetEngineInfo();

            /**
             * @brief [EN] Gets crashed thread ID [ZH] 获取崩溃线程ID
             * @return [EN] Thread ID of the crashed thread, -1 if failed [ZH] 崩溃线程的ID，失败返回-1
             * @note [EN] Not available on PS4/PS5/Switch/Linux [ZH] PS4/PS5/Switch/Linux平台不可用
             */ 
            static long GetCrashThreadId();

            /************************************Android、iOS、Mac端接口************************************/

            /**
             * @brief [EN] Configures callback types for reports (Android, iOS, Mac) [ZH] 设置各类上报的回调开关(Android、iOS、Mac)
             * @param callback_type [EN] There are currently 5 types of callback represented by 5 digits. The first digit represents crash, the second digit represents anr, 
             *                           the third digit represents u3d c# error, the fourth digit represents js, and the fifth digit represents lua
             *                      [ZH] 目前是5种类型，用5位表示。第一位表示crash，第二位表示anr，第三位表示u3d c# error，
             *                           第四位表示js，第五位表示lua。默认全开
             */ 
            static void ConfigCallbackType(int32_t callback_type);

            /**
             * @brief [EN] Sets device model(Android, iOS, Mac) [ZH] 设置设备型号(Android, iOS, Mac)
             * @param device_model
             */
            static void SetDeviceModel(const char* device_model);

             /**
             * @brief [EN] Actively reports errors with JSON format (Android, iOS, Mac) [ZH] 以JSON格式主动上报错误 (Android, iOS, Mac)
             * @param type [EN] Exception type, 0 ~ 3 are internal reserved types and are invalid if passed in, C#: 4, js: 5, lua: 6
             * @param exception_name [EN] Exception Name, cannot be null [ZH] 异常名称，不能为null
             * @param exception_msg [EN] Exception reason, cannot be null [ZH] 异常原因，不能为null
             * @param exception_stack [EN] Stack trace, cannot be null [ZH] 堆栈，不能为null
             * @param params_json [EN] JSON string of serialized map [ZH] map序列化后的JSON字符串
             * @param dump_native_type [EN] 0: close, 1: call system interface dump, 3: minidump (only valid for mobile)
             *                         [ZH] 0：关闭，1：调用系统接口dump，3：minidump(仅移动端有效)
             * @param errorAttachmentPath [EN] Absolute path of the attachment [ZH] 附件绝对路径
             * 
             */
            static void ReportExceptionJson(int type, const char* exception_name, const char* exception_msg,
                                            const char* exception_stack, const char* params_json, int dump_native_type = 0, const char* errorAttachmentPath = "");

             /**
             * @brief [EN] Reports lightweight log (Android, iOS, Mac, Harmony) [ZH] 上报轻量级日志(Android、iOS、Mac, Harmony)
             * @param msg_type [EN] Message type [ZH] 消息类型
             * @param msg [EN] Message content [ZH] 消息内容
             */
            static void ReportLogInfo(const char* msg_type, const char* msg);

            /**
             * @brief [EN] Sets the scene field name(Android、iOS、Mac) [ZH] 设置场景字段(Android、iOS、Mac)
             * @param sceneId
             * @param upload [EN] Whether to upload scene change [ZH] 是否上报场景变更
             * @note [EN] Each network connection or report can carry this field to calculate crash rates and other data for different scenes.
             *            It is not available on PS4、PS5、Switch and Linux.
             *       [ZH] 每一个联网或者上报都可以携带该字段，实现针对不同的子场景计算崩溃率等数据。 PS4、PS5、Switch、Linux暂不可用
             */
            static void SetScene(const char *sceneId, bool upload = false);

            /**
             * @brief [EN] Sets the scene field name(Android、iOS、Mac) [ZH] 设置场景字段(Android、iOS、Mac)
             * @param sceneId
             * @param upload [EN] Whether to upload scene change [ZH] 是否上报场景变更
             * @note [EN] Each network connection or report can carry this field to calculate crash rates and other data for different scenes.
             *            It is not available on PS4、PS5、Switch and Linux.
             *       [ZH] 每一个联网或者上报都可以携带该字段，实现针对不同的子场景计算崩溃率等数据。 PS4、PS5、Switch、Linux暂不可用
             */
            static void SetScene(int sceneId, bool upload = false);

            /**
             * @brief [EN] Enable multi-signal capture (Android、iOS、Mac) [ZH] 开启多信号捕获 (Android、iOS、Mac)
             * @param enable
             * @note [EN] Disabled by default. Capture multiple signals from different threads and upload information about the first signal.
             *       [ZH] 默认关闭，捕捉来自不同线程的多个信号，并上传第一个信号的信息。
             */
            static void SetCatchMultiSignal(bool enable);

            /**
             * @brief [EN] Enable report last frame (Android、iOS、Mac) [ZH] 开启上报最后一帧(Android、iOS、Mac)
             * @param enable
             * @note [EN] Disabled by default. Goes back up to 256 frames and reports the last frame even if the stack string is full.
             *       [ZH] 默认关闭，最多回溯256帧，即使栈字符串已满，也会上报最后一帧。
             */
            static void SetUnwindExtraStack(bool enable);

            /**
             * @brief [EN] Sets customized device ID (Android、iOS、Mac) [ZH] 设置自定义device ID(Android、iOS、Mac)
             * @param device_id
             */
            static void SetCustomizedDeviceID(const char* device_id);

            /**
             * @brief [EN] Gets SDK defined device ID (Android、iOS、Mac) [ZH] 获取SDK生成的device ID (Android、iOS、Mac)
             * @param data [EN] Pointer to memory for storing device ID [ZH] 指向用于存放device ID的内存
             * @param len [EN] Length of data buffer [ZH] 数据缓冲区长度
             */
            static void GetSDKDefinedDeviceID(void* data, int len);

            /**
             * @brief [EN] Set custom match ID (Android、iOS、Mac) [ZH] 设置自定义match ID (Android、iOS、Mac)
             * @param match_id  Can be used to find crashes and errors in Advanced Search.
             */
            static void SetCustomizedMatchID(const char* match_id);

            /**
             * @brief [EN] Get SDK-generated session ID (Android、iOS、Mac) [ZH] 获取SDK生成的session ID (Android、iOS、Mac)
             * @param data [EN] Pointer to memory for storing session ID [ZH] 指向用于存放session ID的内存
             * @param len [EN] Length of data buffer [ZH] 数据缓冲区长度
             */
            static void GetSDKSessionID(void* data, int len);

            /**
             * @brief [EN] Checks if last time was crash exit (Android、iOS、Mac) [ZH] 检查上次是否为崩溃退出 (Android、iOS、Mac)
             * @note [EN] Generally used in callbacks [ZH] 一般在回调中使用
             * 
             */
            static bool IsLastSessionCrash();

            /**
             * @brief [EN] Get last user ID (Android、iOS、Mac) [ZH] 获取上次的用户ID (Android、iOS、Mac)
             * @param data [EN] Pointer to memory for storing user ID [ZH] 指向用于存放用户ID的内存
             * @param len [EN] Length of data buffer [ZH] 数据缓冲区长度
             * @note [EN] Generally  used with "IsLastSessionCrash" to get the last user ID in a callback 
             *       [ZH] 一般与 "IsLastSessionCrash" 一起使用，在回调中获取上次的用户ID
             */
            static void GetLastSessionUserId(void* data, int len);

            /**
             * @brief [EN] Sets upload thread count (Android、iOS、Mac) [ZH] 设置上报线程数 (Android、iOS、Mac)
             * @param num [EN] Thread count, default is 3 [ZH] 线程数量，默认为3
             */
            static void SetUploadThreadNum(int num);

            /**
             * @brief [EN] Tests OOM crash (Android、iOS、Mac, Harmony) [ZH] 测试OOM崩溃 (Android、iOS、Mac, Harmony)
             */
            static void TestOomCrash();

            /**
             * @brief [EN] Sets game engine type (Android、iOS、Mac) [ZH] 设置游戏引擎类型 (Android、iOS、Mac)
             * @param game_type [EN] 1=COCOS, 2=UNITY, 3=UNREAL [ZH] 1=COCOS, 2=UNITY, 3=UNREAL
             */
            static void SetGameType(int game_type);

            /**
             * @brief [EN] Tests Java crash (Android) [ZH] 测试Java崩溃 (Android)
             */
            static void TestJavaCrash();

            /**
             * @brief [EN] Tests ANR (Application Not Responding, Android) [ZH] 测试ANR(应用无响应, Android)
             */
            static void TestANR();

            /**
             * @brief [EN] Gets crash UUID (Android、iOS、Mac) [ZH] 获取崩溃UUID (Android、iOS、Mac)
             * @param data [EN] Pointer to memory for storing UUID [ZH] 指向用于存放UUID的内存
             * @param len [EN] Length of data buffer [ZH] 数据缓冲区长度
             * @note [EN] Get the UUID of the current report, the UUID is used to uniquely identify a report, usually used in callbacks.
             *       [ZH] 获取当次上报的UUID，该UUID用于唯一标识一次上报，一般在回调中使用
             */
            static void GetCrashUuid(void* data, int len);

            /** 
            * @brief [EN] Sets the logcat buffer size (Android) [ZH] 设置logcat缓存大小 (Android)
            * @param size [EN] Buffer size in KB [ZH] logcat缓存大小(单位：KB)
            * @note [EN] Default is 10KB in non-debug mode, 128KB in debug mode
            *       [ZH] 默认非debug模式下10KB，debug模式下128KB
            */
            static void SetLogcatBufferSize(int size);

            /**
             * @brief [EN] Tests Objective-C crash [ZH] 测试objective-C崩溃(iOS、Mac)
             */
            static void TestOcCrash();

            /**
             * @brief [EN] Starts scheduled dump routine [ZH] 启动定时dump任务
             * @param dumpMode [EN] 1=dump, 2=minidump [ZH] 1=dump, 2=minidump
             * @param startTimeMode [EN] 0=absolute time, 1=relative time (milliseconds) [ZH] 0=绝对时间,1=相对时间(毫秒)
             * @param startTime [EN] Start time [ZH] 启动时间
             * @param dumpInterval [EN] Dump interval (milliseconds) [ZH] dump间隔(毫秒)
             * @param dumpTimes [EN] Dump count [ZH] dump次数
             * @param saveLocal [EN] Whether to save locally [ZH] 是否保存本地
             * @param savePath [EN] Local save path [ZH] 本地保存路径
             */
            static void StartDumpRoutine(int dumpMode, int startTimeMode, long startTime, long dumpInterval,
                                         int dumpTimes, bool saveLocal, const char *savePath);

            /**
             * @brief [EN] Starts monitoring file descriptor count (Android) [ZH] 开始FD数量 (Android)
             * @param interval [EN] Scan interval (milliseconds) [ZH] 扫描间隔(毫秒)
             * @param limit [EN] FD count limit, exceeding will trigger a report [ZH] FD数量限制，超过将触发上报
             * @param dumpType [EN] 0=no dump, 1=system dump, 3=minidump [ZH] 0=不上报dump,1=系统接口dump,3=minidump
             */
            static void StartMonitorFdCount(int interval, int limit, int dumpType);

            /**
             * @brief [EN] Gets exception type code (Android、iOS、Mac) [ZH] 获取异常类型编号 (Android、iOS、Mac)
             * @param name [EN] Exception type name (e.g. "c#", "js", "lua", "custom1") [ZH] 异常类型名(如"c#","js","lua","custom1")
             * @return [EN] Exception type code for ReportException [ZH] 用于ReportException接口的异常类型编号
             */
            static int GetExceptionType(const char *name);

            /**
             * @brief [EN] Tests use-after-free crash (Android) [ZH] 测试释放后使用内存崩溃 (Android)
             */
            static void TestUseAfterFree();

            /**
             * @brief [EN] Enables/disables getting package info (Android) [ZH] 开启/关闭获取PackageInfo (Android)
             * @param enable
             * @note [EN] Enabled by default. Only turn off when encountering compliance issues. Disabling may affect version auto-detection. [ZH] 默认开启。仅在遇到合规问题时关闭。关闭可能导致无法自动获取版本号。
             */
            static void setEnableGetPackageInfo(bool enable);

             /**
             * @brief [EN] Enables detailed page tracing (Android、iOS) [ZH] 开启详细页面追踪 (Android、iOS)s
             * @param enable
             * @note [EN] Disabled by default. Requires CrashSight_AndroidX_release.jar on Android. [ZH] 默认关。安卓需要导入CrashSight_AndroidX_release.jar。
             */
            static void enableDetailedPageTracing(bool enable);

            /**
             * @brief [EN] Checks FD count [ZH] 检查文件描述符数量
             * @param limit [EN] FD count limit [ZH] FD数量限制
             * @param dumpType [EN] 0=no dump, 1=system dump, 3=minidump [ZH] 0=不上报dump,1=系统接口dump,3=minidump
             * @param upload [EN] Whether to upload [ZH] 是否上报
             * @return [EN] true if FD count exceeds limit, false otherwise [ZH] FD数量超过限制返回true，否则返回false
             */
            static bool checkFdCount(int limit, int dumpType, bool upload);

            /**
             * @brief [EN] Sets OOM log path (Android、iOS、Harmony) [ZH] 设置OOM日志路径 (Android、iOS、Harmony)
             * @param log_path [EN] Log file path [ZH] 日志文件路径
             */
            static void SetOomLogPath(const char* log_path);

            /**
             * @brief [EN] Close Crash monitoring (Android、iOS) [ZH] 关闭异常监控 (Android、iOS)
             */ 
            static void CloseCrashReport();

            /**
             * @brief [EN] Start Crash monitoring (Android、iOS) [ZH] 开启异常监控 (Android、iOS)
             */ 
            static void StartCrashReport();

            /**
             * @brief [EN] Restart Crash monitoring (Android、iOS) [ZH] 重启异常监控 (Android、iOS)
             */ 
            static void RestartCrashReport();

            /**
             * @brief [EN] Reports jank information [ZH] 上报卡顿
             * @param type [EN] Jank type (reserved for future use) [ZH] 卡顿类型(保留字段，暂未使用)
             * @param exceptionName [EN] Exception name [ZH] 异常名称
             * @param exceptionMsg [EN] Exception message [ZH] 异常消息
             * @param exceptionStack [EN] Stack trace [ZH] 堆栈信息
             * @param extInfoJsonStr [EN] Extra info in JSON [ZH] 额外信息的JSON字符串
             * @param reportInfoOption [EN] Collection options for jank reporting information, represented by 5 binary bits for different options, 0 indicating no collection of additional information.
             *                              The 1st bit: Collect system logs, the 2nd bit: Collect custom logs, the 3rd bit: Collect user-defined key-value pairs, 
             *                              the 4th bit: Trigger user callbacks, the 5th bit: Automatically dump stack traces.
             *                         [ZH] jank上报信息收集选项，用5位二进制位表示不同选项，0表示不收集任何额外信息
             *                              第1位:收集系统日志，第2位:收集自定义日志，第3位:收集用户自定义键值对，第4位:触发用户回调，第5位:自动dump堆栈
             * @param jankAttachmentPath [EN] Attachment path [ZH] 附件路径
             */
            static void ReportJank(int type, const char* exceptionName,
                    const char* exceptionMsg, const char* exceptionStack,
                    const char* extInfoJsonStr, int reportInfoOption, const char* jankAttachmentPath = nullptr);

            static void ReportStuck(int threadId, int maxChecks, long checkInterval,
                    const char *name, const char *message, const char *extraInfo,
                    int dumpNativeType, const char *attachPath);

            /************************************PC、Xbox端接口************************************/

            /**
             * @brief [EN] Sets VEH(Vectored Exception Handling) enable status (Win, Xbox)
             *        [ZH] 设置VEH(向量化异常处理)捕获开关(Win、Xbox)
             * @param enable
             */
            static void SetVehEnable(bool enable);

            /**
             * @brief [EN] Manually reports a crash (Win, Xbox) [ZH] 主动上报崩溃信息(Win、Xbox)
             */
            static void ReportCrash();

            /**
             * @brief [EN] Manually reports dump file (Win, Xbox) [ZH] 主动上报dump文件(Win、Xbox)
             * @param dump_path [EN] Dump file directory [ZH] dump文件目录
             * @param is_async [EN] Whether to report asynchronously [ZH] 是否异步上报
             */
            static void ReportDump(const char* dump_path, bool is_async);

            /**
             * @brief [EN] Sets crash callback (Win, Xbox) [ZH] 设置崩溃回调(Win、Xbox)
             * @param callback [EN] Crash callback function pointer [ZH] 崩溃回调函数指针
             */
            static void SetCrashCallback(CrashCallbackFuncPtr callback);

            /**
             * @brief [EN] Sets extra exception handler (Win, Xbox) [ZH] 设置额外异常捕获的开关(Win、Xbox)
             * @param extra_handle_enable
             */
            static void SetExtraHandler(bool extra_handle_enable);

            /**
             * @brief [EN] Uploads dump file from given path (Win, Xbox) [ZH] 上传指定路径的dump文件(Win、Xbox)
             * @param dump_dir [EN] Dump file directory [ZH] dump文件目录
             * @param is_extra_check [EN] Extra validation check, Default is false [ZH] 额外校验检查, 默认填false即可
             */
            static void UploadGivenPathDump(const char* dump_dir, bool is_extra_check);

            /**
             * @brief [EN] Enables/disables Unreal Critical Error reporting (Win, Xbox) 
             *        [ZH] 开启/关闭 UE Critical Error上报(Win、Xbox)
             * @param enable
             */
            static void UnrealCriticalErrorEnable(bool enable);

            /**
             * @brief [EN] Enable Report Only the First Crash (Win, Xbox) [ZH] 仅上报首条崩溃的开关(Win、Xbox)
             * @param enable
             * @note [EN] Only takes effect after VEH capture is enabled [ZH] 开启VEH捕获后才生效
             */
            static void OnlyUploadFirstCrash(bool enable);

            /**
             * @brief [EN] Sets dump type (Win, Xbox) [ZH] 设置dump类型(Win、Xbox)
             * @param dump_type [EN] Dump type, see Windows official definition [ZH] dump类型，详见Windows官方定义
             */
            static void SetDumpType(int dump_type);

            /**
             * @brief [EN] Adds valid exception code (Win, Xbox) [ZH] 添加可用的异常类型(Win、Xbox)
             * @param exp_code [EN] Exception code, see Windows official definition [ZH] 异常代码，详见Windows官方定义
             * @note [EN] Please consult CrashSight developers before using [ZH] 使用前请咨询CrashSight开发人员
             */
            static void AddValidExpCode(unsigned long exp_code);

            /**
             * @brief [EN] Uploads crash with GUID (Win, Xbox) [ZH] 根据问题GUID上报(Win、Xbox)
             * @param guid [EN] GUID identifying the specific issue, an be obtained through callback. [ZH] 指向唯一问题的代码，可通过回调获取
             */
            static void UploadCrashWithGuid(const char *guid);

            /**
             * @brief [EN] Sets crash upload enable status (Win, Xbox) [ZH] 设置崩溃上报开关(Win、Xbox)
             * @param enable
             */
            static void SetCrashUploadEnable(bool enable);

            /**
             * @brief [EN] Sets workspace (Win, Xbox) [ZH] 设置工作空间(Win、Xbox)
             * @param workspace [EN] Absolute path of workspace [ZH] 工作空间的绝对路径
             */
            static void SetWorkSpace(const char *workspace);

            /**
             * @brief [EN] Sets workspace (wide char version, Win, Xbox) [ZH] 设置工作空间(宽字符版本，Win、Xbox)
             * @param workspace [EN] Absolute path of workspace [ZH] 工作空间的绝对路径
             */
            static void SetWorkSpaceW(const wchar_t *workspace);

            /**
             * @brief [EN] Sets custom attachment path (Not available on Switch) (Win, Xbox) [ZH] 设置自定义附件路径(Switch不可用)(Win、Xbox)
             * @param attach_path [EN] Absolute path to attachment directory [ZH] 附件的绝对路径
             */
            static void SetAttachPath(const char* attach_path);

            /**
             * @brief [EN] Sets custom attachment path (wide char version, Not available on Switch) (Win, Xbox) [ZH] 设置自定义附件路径(宽字符版本，Switch不可用)(Win、Xbox)
             * @param attach_path [EN] Absolute path to attachment directory [ZH] 附件的绝对路径
             */
            static void SetAttachPathW(const wchar_t* attach_path);

            /************************************PS4、PS5、Switch、Linux端接口************************************/

            /**
             * @brief [EN] Sets error upload interval (PS4, PS5, Switch) [ZH] 设置错误上报间隔(PS4、PS5、Switch)
             * @param interval [EN] Error upload interval in seconds [ZH] 错误上报间隔(单位：秒)
             */
            static void SetErrorUploadInterval(int interval);

            /**
             * @brief [EN] Enables/disables error upload (PS4, PS5, Switch) [ZH] 开启/关闭错误上报(PS4、PS5、Switch)
             * @param enable
             */
            static void SetErrorUploadEnable(bool enable);

            /**
             * @brief [EN] Sets record file directory (Linux) [ZH] 设置所有记录文件的路径(Linux)
             * @param record_dir [EN] Path of record files [ZH] 记录文件的路径
             */
            static void SetRecordFileDir(const char* record_dir);

            /************************************已废弃接口************************************/

            /**
             * @brief [EN] Initialize CrashSight (Deprecated) [ZH] 初始化CrashSight(已废弃)
             * @param user_id [EN] User ID (Win, Xbox) [ZH] 用户ID(Win、Xbox)
             * @param version [EN] Application version [ZH] 应用版本号
             * @param key [EN] AppKey [ZH] AppKey
             * @deprecated [EN] Use InitWithAppId instead [ZH] 请使用InitWithAppId替代
             */
            CS_DEPRECATED("InitContext is deprecated. Use InitWithAppId instead")
            static int InitContext(const char* user_id, const char* version, const char* key);

            /**
             * @brief [EN] Initialize CrashSight (Deprecated) [ZH] 初始化CrashSight(已废弃)
             * @param app_id
             * @param app_key
             * @param app_version [EN] Application version [ZH] 应用版本号
             * @deprecated [EN] Use InitWithAppId instead [ZH] 请使用InitWithAppId替代
             */
            CS_DEPRECATED("Init is deprecated. Use InitWithAppId instead")
            static void Init(const char* app_id, const char* app_key, const char* app_version);

            /**
             * @brief [EN] Enable/Disable crash monitoring (Deprecated) [ZH] 开启/关闭崩溃捕获(已废弃)
             * @param enable
             * @deprecated [EN] Deprecated since version 2.1.1 [ZH] 自2.1.1版本起废弃
             */
            CS_DEPRECATED("MonitorEnable is deprecated since version 2.1.1")
            static void MonitorEnable(bool enable);

            /**
             * @brief [EN] Configure default settings before initialization (Deprecated) [ZH] 初始化前填写配置信息(已废弃)
             * @param channel [EN] Channel (Deprecated, use empty string "") [ZH] 渠道(已废弃，默认填"")
             * @param version [EN] Application version [ZH] 应用版本号
             * @param user [EN] User ID [ZH] 用户ID
             * @param delay [EN] Delay time [ZH] 延时时间
             * @deprecated [EN] Use SetUserId and SetAppVersion instead [ZH] 请使用SetUserId和SetAppVersion替代
             */
            CS_DEPRECATED("ConfigDefault is deprecated. Use SetUserId and SetAppVersion instead")
            static void ConfigDefault(const char* channel, const char* version, const char* user, long delay);

            CS_DEPRECATED("GetPlatformCode is deprecated.")
            static int GetPlatformCode();
        };
    }  // namespace CrashSight
}  // namespace GCloud
